/*
 * KontaFirm.java
 *
 * Created on 25 marzec 2004, 19:50
 */

package serwer.konto;


import java.lang.String;
import net.sf.hibernate.*;
import java.rmi.server.UnicastRemoteObject;
import java.awt.image.BufferedImage;
import java.rmi.RemoteException;
import java.awt.Point;
import java.util.Date;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.TreeSet;
import java.io.*;

import serwer.Jadro;
import serwer.Hibcio;
import shared.*;
import serwer.mail.*;
import serwer.mapa.*;

/** KontaFirm dostarczaj wszystkich potrzebnych operacji do administrowania
 * kontami firm.
 *
 * @author  ula
 */
public class KontaFirm
extends UnicastRemoteObject implements InterfejsKontaFirm {
    
    /** Zwraca nowy obiekt klasy KontaFirm.
     *  @throws RemoteException przy bdzie inicjalizacji interfejsu RMI
     */
    public KontaFirm() throws java.rmi.RemoteException {
        super();
    }
    
    /** Ustala miejsce ogoszenia w rankingu firm. */
    private int ustalRanking(Ogloszenie ogloszenie) {
        int rank = 0;
        
        //preferujemy platne
        OgloszeniePlatne platne = ogloszenie.getPlatne();
        if (platne != null) {
            rank += 100;
            if (platne.getGodziny() != null)
                rank += 10;
            if (platne.getSlowoKluczowe1() != null)
                rank += 10;
            if (platne.getSlowoKluczowe2() != null)
                rank += 10;
            if (platne.getSlowoKluczowe3() != null)
                rank += 10;
            if (platne.getSlowoKluczowe4() != null)
                rank += 10;
            if (platne.getUslugi() != null)
                rank += 10;
        }
        //za kazde pozostale wypelnione pole +10
        if (ogloszenie.getFax() != null)
            rank += 10;
        if (ogloszenie.getStronaInternetowa() != null)
            rank += 10;
        if (ogloszenie.getEmail() != null)
            rank += 10;
        if (ogloszenie.getUwagi() != null)
            rank += 10;
        
        return rank;
    }
    
    /** Tworzy nowe konto dla firmy na podstawie ogoszenia. Ogoszenie umieszcza
     * w bazie danych. 
     *
     * @param ogloszenie    ogoszenie, ktre firma wanie stworzya
     * @param telefony      dodatkowe telefony do firmy, kady telefon jest list 
     *                      skadajc si z dwch elementw: numeru telefonu i 
     *                      jego opisu
     * @return napis z opisem ewentualnego bdu
     *
     * @see KontaFirm#modyfikujOgloszenie
     *
     * @throws Wyjatek gdy nie uda si utworzy konta dla danego ogoszenia
     * @author ula, ek197882
     */
    public String utworzKonto(Ogloszenie ogloszenie, ArrayList telefony) throws Wyjatek {
        Session s = null;
        Transaction tx = null;
        //przygotowanie do wstawienia
        //1. spr nru domu, czy istnieje
        shared.Ulica ul = ogloszenie.getUlica();
        ZarzUlica zarzUl = Jadro.ulice.zarzUlica(ul);
        Point wspDomu;
        try {
            wspDomu = zarzUl.wspolrzedneObiektu(ogloszenie.getNrDomu());
        } catch (Exception e) {
            e.printStackTrace();
            throw new Wyjatek();
        }
        if (wspDomu == null)
            return "Niepoprawny numer domu.\n";
        
        //2. ustawienie popularnosci na zero
        ogloszenie.setPopularnosc(0);
        //3. wyliczenie i ustawienie wspolrzednych
        ogloszenie.setWspolrzedna_x((int)wspDomu.getX());
        ogloszenie.setWspolrzedna_y((int)wspDomu.getY());
        //4. ustawienie daty aktualizacji, na razie nieistotna wstawiamy dzisiejsza
        ogloszenie.setDataAktualizacji(new Date());
        //5. ogl jeszcze nie jest zaakceptowane
        ogloszenie.setZaakceptowane(false);
        //spr czy platne i jesli tak to ustawiamy odp pola w platnym
        OgloszeniePlatne ogP = ogloszenie.getPlatne();
        if (ogP != null) {
            //data oplacenia na dzisiejsza, bez znaczenia bo jeszcze ogl niezaakceptowane
            ogP.setDataOplacenia(new Date());
        }
        
        //wszystko ustawione, wyliczamy ranking
        ogloszenie.setRanking(this.ustalRanking(ogloszenie));
        
        
        //sprawdzamy czy w bazie nie ma ogloszenia o tym samym nipie i os prawnej
        //boolean jest = this.czyJestOgloszenie(new Integer(ogloszenie.getNip()), new Boolean(ogloszenie.isOsobowoscPrawna()));
        Integer id = this.czyJestOgloszenie(new Long(ogloszenie.getNip()), new Boolean(ogloszenie.isOsobowoscPrawna()));
        //if (jest)
        if (id != null)
            return "Ogoszenie o podanym nipie jest ju w bazie.\n";
        
        
        //wstawiamy ogloszenie do bazy, a potem tel dla tego ogloszenia
        try {
            s = Hibcio.openSession();
            tx = s.beginTransaction();
            if (ogloszenie.getPlatne() != null)
                s.saveOrUpdate(ogloszenie.getPlatne());
            s.saveOrUpdate(ogloszenie);
            //wstawilismy ogloszenie teraz wszystkie tel dla niego
            Iterator iter = telefony.iterator();
            while (iter.hasNext()) {
                ArrayList l = (ArrayList) iter.next();
                String numer = (String)l.get(0);
                String opis = (String)l.get(1);
                Telefon tel = new Telefon(opis, numer, ogloszenie);
                s.saveOrUpdate(tel);
            }
            tx.commit();
        } catch (Exception e) {
            if (tx != null)
                try {
                    tx.rollback();
                } catch (HibernateException he) {
                    throw new Wyjatek();
                }
            e.printStackTrace();
            throw new Wyjatek();
        } finally {
            Hibcio.closeSession(s);
        }
        
        return "OK";
    }

    /** Modyfikuje istniejce w bazie ogoszenie firmy.
     *
     * @param ogloszenie    ogoszenie, ktre jest modyfikowane
     * @param telefony      dodatkowe telefony do firmy
     * @return napis z opisem ewentualnego bdu
     * @throws Wyjatek gdy nie uda si zmodyfikowa danego ogoszenia
     * 
     * @see KontaFirm#utworzKonto
     *
     * @author ula, ek197882
     */
    public String modyfikujOgloszenie(Ogloszenie ogloszenie, ArrayList telefony) throws Wyjatek {
        Session s = null;
        Transaction tx = null;
        
        //1. spr poprawnosci modyfikacji
        //1a. czy jest nowy nr domu
        shared.Ulica ul = ogloszenie.getUlica();
        ZarzUlica zarzUl = Jadro.ulice.zarzUlica(ul);
        Point wspDomu;
        try {
            wspDomu = zarzUl.wspolrzedneObiektu(ogloszenie.getNrDomu());
        } catch (Exception e) {
            e.printStackTrace();
            throw new Wyjatek();
        }
        if (wspDomu == null)
            return "Niepoprawny numer domu.\n";
        ogloszenie.setWspolrzedna_x((int)wspDomu.getX());
        ogloszenie.setWspolrzedna_y((int)wspDomu.getY());
        
        //1b. czy nip nie zostal zmieniony na nip innej firmy z naszej bazy
        Integer id = czyJestOgloszenie(new Long(ogloszenie.getNip()), new Boolean(ogloszenie.isOsobowoscPrawna()));
        if (id != null && !id.equals(ogloszenie.getId()))
            return "Og\u0142oszenie o podanym nipie istnieje ju w bazie.\n Nie mona zmodyfikowa og\u0142oszenia.";
        
        //2. ustawiamy nowe parametry ogloszenia
        ogloszenie.setDataAktualizacji(new Date());
        ogloszenie.setZaakceptowane(false);
        ogloszenie.setRanking(this.ustalRanking(ogloszenie));
        
        //3. pobramy stare ogloszenie
        Ogloszenie stare = pobierzOgloszenie(ogloszenie.getId());
        
        //4. zatwierdzamy zmiany w ogloszeniu
        try {
            s = Hibcio.openSession();
            tx = s.beginTransaction();
            
            //4a. platne => bezplatne
            if (stare.getPlatne() != null && ogloszenie.getPlatne() == null) {
                s.delete(stare.getPlatne());
                usunLogo(ogloszenie.getId());
            }
            //4b. bezplatne => platne
            else if (stare.getPlatne() == null && ogloszenie.getPlatne() != null) {
                s.saveOrUpdate(ogloszenie.getPlatne());
            }
            
            //4c. platne => platne
            else if (stare.getPlatne() != null && ogloszenie.getPlatne() != null)
                s.update(ogloszenie.getPlatne());
            
            //5. modyfikujemy ranking
            ogloszenie.setRanking(this.ustalRanking(ogloszenie));
            
            s.update(ogloszenie);
            
            //6. zatwierdzamy zmiany w telefonach
            int u = s.delete("from shared.Telefon where ogloszenie = " + ogloszenie.getId());
            Iterator iter = telefony.iterator();
            while (iter.hasNext()) {
                ArrayList l = (ArrayList) iter.next();
                String numer = (String)l.get(0);
                String opis = (String)l.get(1);
                Telefon tel = new Telefon(opis, numer, ogloszenie);
                s.saveOrUpdate(tel);
            }
            
        } catch (Exception e) {
            if (tx != null)
                try {
                    tx.rollback();
                } catch (HibernateException he) {
                    throw new Wyjatek();
                }
            e.printStackTrace();
            throw new Wyjatek();
        } finally {
            Hibcio.closeSession(s);
        }
        
        return "OK";
    }
    
        /** Modyfikuje istniejce w bazie ogoszenie firmy. Metoda dla administratora.
     *
     * @param ogloszenie    ogoszenie, ktre jest modyfikowane
     * @param telefony      dodatkowe telefony do firmy
     * @return napis z opisem ewentualnego bdu
     * @throws Wyjatek gdy nie uda si zmodyfikowa danego ogoszenia
     * 
     * @see KontaFirm#utworzKonto
     *
     * @author ula, ek197882
     */
    public String modyfikujOgloszenieAdmin(Ogloszenie ogloszenie, ArrayList telefony) throws Wyjatek {
        Session s = null;
        Transaction tx = null;
        
        //1. spr poprawnosci modyfikacji
        //1a. czy jest nowy nr domu
        shared.Ulica ul = ogloszenie.getUlica();
        ZarzUlica zarzUl = Jadro.ulice.zarzUlica(ul);
        Point wspDomu;
        try {
            wspDomu = zarzUl.wspolrzedneObiektu(ogloszenie.getNrDomu());
        } catch (Exception e) {
            e.printStackTrace();
            throw new Wyjatek();
        }
        if (wspDomu == null)
            return "Niepoprawny numer domu.\n";
        ogloszenie.setWspolrzedna_x((int)wspDomu.getX());
        ogloszenie.setWspolrzedna_y((int)wspDomu.getY());
        
        //1b. czy nip nie zostal zmieniony na nip innej firmy z naszej bazy
        Integer id = czyJestOgloszenie(new Long(ogloszenie.getNip()), new Boolean(ogloszenie.isOsobowoscPrawna()));
        if (id != null && !id.equals(ogloszenie.getId()))
            return "Og\u0142oszenie o podanym nipie istnieje ju w bazie.\n Nie mona zmodyfikowa og\u0142oszenia.";
        
        //2. ustawiamy nowe parametry ogloszenia
        ogloszenie.setDataAktualizacji(new Date());
        //ogloszenie.setZaakceptowane(true);
        ogloszenie.setRanking(this.ustalRanking(ogloszenie));
        
        //3. pobramy stare ogloszenie
        Ogloszenie stare = pobierzOgloszenie(ogloszenie.getId());
        
        //4. zatwierdzamy zmiany w ogloszeniu
        try {
            s = Hibcio.openSession();
            tx = s.beginTransaction();
            
            //4a. platne => bezplatne
            if (stare.getPlatne() != null && ogloszenie.getPlatne() == null) {
                s.delete(stare.getPlatne());
                usunLogo(ogloszenie.getId());
            }
            //4b. bezplatne => platne
            else if (stare.getPlatne() == null && ogloszenie.getPlatne() != null) {
                s.saveOrUpdate(ogloszenie.getPlatne());
            }
            
            //4c. platne => platne
            else if (stare.getPlatne() != null && ogloszenie.getPlatne() != null)
                s.update(ogloszenie.getPlatne());
            
            //5. modyfikujemy ranking
            ogloszenie.setRanking(this.ustalRanking(ogloszenie));
            
            s.update(ogloszenie);
            
            //6. zatwierdzamy zmiany w telefonach
            int u = s.delete("from shared.Telefon where ogloszenie = " + ogloszenie.getId());
            Iterator iter = telefony.iterator();
            while (iter.hasNext()) {
                ArrayList l = (ArrayList) iter.next();
                String numer = (String)l.get(0);
                String opis = (String)l.get(1);
                Telefon tel = new Telefon(opis, numer, ogloszenie);
                s.saveOrUpdate(tel);
            }
            
        } catch (Exception e) {
            if (tx != null)
                try {
                    tx.rollback();
                } catch (HibernateException he) {
                    throw new Wyjatek();
                }
            e.printStackTrace();
            throw new Wyjatek();
        } finally {
            Hibcio.closeSession(s);
        }
        
        return "OK";
    }
    
    /* Czy jest takie ogloszenie w bazie. */
    private Integer czyJestOgloszenie(Long nip, Boolean os) throws Wyjatek {
        Ogloszenie o = new Ogloszenie();
        
        Session s = null;
        Query q;
        
        try {
            s = Hibcio.openSession();
            //znajdz odpowiednie ogloszenie
            q = s.createQuery("from shared.Ogloszenie as og " +
            "where og.nip = :nip " +
            "and og.osobowoscPrawna = :os ");
            q.setParameter("nip", nip);
            q.setParameter("os", os);
            o = (Ogloszenie) q.uniqueResult();
        } catch (Exception e) {
            e.printStackTrace();
            throw new Wyjatek();
        } finally {
            Hibcio.closeSession(s);
        }
        if (o == null)
            //return false;
            return null;
        else
            //return true;
            return o.getId();
    }
    
    
    /** Zwraca ogoszenie firmy na podstawie podanego id.
     * @param id    id firmy, ktrej ogoszenie naley zwrci
     * @return ogoszenie danej firmy
     *@throws Wyjatek gdy nie uda si pobra ogoszenia
     */
    public Ogloszenie pobierzOgloszenie(Integer id) throws Wyjatek {
        Ogloszenie o = new Ogloszenie();
        
        Session s = null;
        Query q;
        
        try {
            s = Hibcio.openSession();
            //znajdz odpowiednie ogloszenie
            //q = s.createQuery("from shared.Ogloszenie as og " +
            //"where og.id = :id ");
            //q.setParameter("id", id);
            //o = (Ogloszenie) q.uniqueResult();
            s.load(o, id);
        } catch (Exception e) {
            e.printStackTrace();
            throw new Wyjatek();
        } finally {
            Hibcio.closeSession(s);
        }
        return o;
    }
    
    /** Zwraca telefony zwizane z dan firm.
     * @param id    id firmy
     * @return      zbir telefonw
     *@throws Wyjatek gdy nie powiod si operacje na bazie danych
     */
    public ArrayList pobierzTelefony(Integer id) throws Wyjatek {
        ArrayList tset = new ArrayList();
        int ileTel = 0;
        Session s = null;
        Query q;
        
        try {
            s = Hibcio.openSession();
            q = s.createQuery("from shared.Telefon as tel " +
            "where tel.ogloszenie.id = :id");
            q.setParameter("id", id);
            Iterator iter = q.iterate();
            while (iter.hasNext()) {
                Telefon tel = (Telefon) iter.next();
                ArrayList l = new ArrayList(2);
                l.add(0, tel.getNr());
                l.add(1, tel.getOpis());
                tset.add(ileTel, l);
                ileTel++;
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new Wyjatek();
        } finally {
            Hibcio.closeSession(s);
        }
        return tset;
    }
    
    /** Sprawdza poprawno hasa.
     *
     * @param nip   jest loginem do danego konta
     * @param haslo haso poddawane autoryzacji
     * @return ogoszenie o podanym nipie
     * @throws Wyjatek gdy nie uda si dokona autoryzacji
     */
    public Ogloszenie autoryzacja(Nip nip, String haslo) throws Wyjatek {
        Ogloszenie o = new Ogloszenie();
        
        Session s = null;
        Query q;
        
        try {
            s = Hibcio.openSession();
            //znajdz odpowiednie ogloszenie
            q = s.createQuery("from shared.Ogloszenie as og " +
            "where og.nip = :nip " +
            " and og.osobowoscPrawna = :osp ");
            q.setParameter("nip", new Long(nip.nipToInt()));
            q.setParameter("osp", new Boolean(nip.czyOsPrawna()));
            o = (Ogloszenie) q.uniqueResult();
        } catch (Exception e) {
            throw new Wyjatek();
        } finally {
            Hibcio.closeSession(s);
        }
        if (o == null || (o.getHaslo().compareTo(haslo)) != 0) {
            return null;
        } else
            return o;
    }
    
    /** 
     * Wykonuje procedur przypominania hasa na podstawie NIP.
     * Haso zostanie wysane na podany przez firm adres kontaktowy 
     * e-mail.
     *
     * @param nip   nip, po ktrym bdziemy identyfikowa firm
     * @throws Wyjatek jeli nie uda si odnale ogoszenia firmy w bazie
     */
    public void przypomnijHaslo(Nip nip) throws Wyjatek {
        Ogloszenie o;
        
        Session s = null;
        Query q;
        
        try {
            s = Hibcio.openSession();
            //znajdz odpowiednie ogloszenie
            q = s.createQuery("from shared.Ogloszenie as og " +
            "where og.nip = :nip " +
            " and og.osobowoscPrawna = :osp ");
            q.setParameter("nip", new Long(nip.nipToInt()));
            q.setParameter("osp", new Boolean(nip.czyOsPrawna()));
            o = (Ogloszenie) q.uniqueResult();
        } catch (Exception e) {
            throw new Wyjatek();
        } finally {
            Hibcio.closeSession(s);
        }
        //wyslij zapomniane haslo na podany adres
        try {
            //SendMailUsingAuthentication mail = new SendMailUsingAuthentication();
            Mail mail = new Mail();
            String[] recipients = new String[1];
            recipients[0] = o.getEkontakt();
            String subject = "NIIKT::zapomniane haso";
            String message = "Wiadomo z portalu NIIKT::\nHaso dla firmy o NIP-ie:: \n\t"+
                Nip.intToNip(o.getNip(), o.isOsobowoscPrawna())+"\nto::\n\t"+o.getHaslo()+
                "\n\nPozdrawiamy, \nAdministracja systemu NIIKT";
            mail.postMail(recipients, subject, message, Jadro.getParametr("NIIKT_EMAIL"));
        } catch (Exception e) {
            // e.printStackTrace();
            throw new Wyjatek();
        }
    }
    
    /** 
     * Zmienia haso firmy.
     * 
     * @param id        identyfikator firmy, zakadamy, e taka jest w bazie
     * @param nowe      nowe haso
     * @throws Wyjatek gdy operacje na bazie danych nie powiod si
     */
    public void zmienHaslo(Integer id, String nowe) throws Wyjatek {
        Ogloszenie o;
        
        Session s = null;
        Query q;
        
        try {
            s = Hibcio.openSession();
            //znajdz odpowiednie ogloszenie
            q = s.createQuery("from shared.Ogloszenie as og " +
            "where og.id = :id ");
            q.setParameter("id", id);
            o = (Ogloszenie) q.uniqueResult();
        } catch (Exception e) {
            e.printStackTrace();
            throw new Wyjatek();
        } finally {
            Hibcio.closeSession(s);
        }
        
        //spr czy podane dobre stare haslo
        /*
        if ((o.getHaslo().compareTo(stare)) != 0) {
            return false;
        }
        else {
         **/
        // stare=OK wstawiamy nowe do bazy
        o.setHaslo(nowe);
        try {
            s = Hibcio.openSession();
            s.saveOrUpdate(o);
        } catch (Exception e) {
            e.printStackTrace();
            throw new Wyjatek();
        } finally {
            Hibcio.closeSession(s);
        }
    }

    /**
     * Generuje katalog, w ktrym bdzie si znajdowao logo.
     *
     * @param id Identyfikator firmy
     * @return Skonstruowana cieka
     *
     * @author ek197882
     */
    private String konstruujPoczSciezkiLogo(Integer id) {
        return Jadro.getParametr("NIIKT_WWW") + File.separator +
        "loga" + File.separator + id.toString().charAt(0);
    }
    
    /**
     * Generuje nazw logo firmy.
     *
     * @param id Identyfikator firmy, dla ktrej zostanie skonstruowana nazwa logo
     * @return Skonstruowana cieka
     *
     * @author ek197882
     */
    public String konstruujSciezkeLogo(Integer id) {
        return konstruujPoczSciezkiLogo(id) + File.separator + id.toString();
    }
    
    /**
     * Generuje url do logo firmy. Np. typu http://...serwer.../loga/1/15
     *
     * @param id Identyfikator firmy, dla ktrej zostanie skonstruowana nazwa logo
     * @return Skonstruowana cieka
     *
     * @author ek197882
     */
    public String konstruujHttpSciezkeLogo(Integer id) {
        return Jadro.getParametr("NIIKT_HTTP") + File.separator +
        "loga" + File.separator + id.toString().charAt(0) + File.separator + id.toString();
    }
    
    
    /**
     * Zapisuje ogoszenie na serwer.
     *
     * @param id Identyfikator firmy, ktrej ogoszenie zostanie zapisane na serwerze
     * @param pg Plik graficzny z logo danej firmy
     * @throws Wyjatek gdy nie uda si zapisa logo na serwerze
     *
     * @author ek197882
     */
    public void zapiszLogoFirmy(Integer id, PlikGraficzny pg) throws Wyjatek {
        try {
            File f = new File(konstruujSciezkeLogo(id));
            File dir = new File(konstruujPoczSciezkiLogo(id));
            dir.mkdirs();
            f.createNewFile();
            javax.imageio.ImageIO.write(pg.getBufferedImage(), "jpg", f);
        } catch (Exception e) {
            e.printStackTrace();
            throw new Wyjatek();
        }
    }
    
    /**
     * Wyszukuje id firmy o zadanym nipie i osobowoci prawnej.
     *
     * @param nip    nip
     * @param osobowoscPrawna osobowo prawna danego nip
     * @return liczba reprezentujca wyszukane id
     * @throws Wyjatek gdy operacje na bazie nie powiod si
     *
     * @author ek197882
     */
    public Integer pobierzId(long nip, boolean osobowoscPrawna) throws Wyjatek {
        Ogloszenie o;
        Session s = null;
        Query q;
        
        try {
            s = Hibcio.openSession();
            q = s.createQuery("from shared.Ogloszenie as og " +
            "where og.nip = :nip " +
            " and og.osobowoscPrawna = :osp ");
            q.setParameter("nip", new Long(nip));
            q.setParameter("osp", new Boolean(osobowoscPrawna));
            o = (Ogloszenie) q.uniqueResult();
        } catch (Exception e) {
            throw new Wyjatek();
        } finally {
            Hibcio.closeSession(s);
        }
        
        if (o == null)
            return null;
        
        return o.getId();
    }
    
    /**
     * Tworzy konto i zapisuje logo.
     *
     * @param ogloszenie     nowe ogoszenie
     * @param telefony       lista telefonw dla danego ogoszenia
     * @param pg             plik graficzny z logo
     * @return wynik utworzenia konta jako informacja w stringu
     * @throws Wyjatek gdy nie uda si utworzy nowego ogoszenia w bazie
     *
     * @author ek197882
     */
    public String utworzKontoZLogo(Ogloszenie ogloszenie, ArrayList telefony, PlikGraficzny pg) throws Wyjatek {
        String s = utworzKonto(ogloszenie, telefony);
        try {
            zapiszLogoFirmy(pobierzId(ogloszenie.getNip(), ogloszenie.isOsobowoscPrawna()), pg);
        } catch (Exception e) {
            return "Konto utworzono, lecz logo nie zosta\u0142o zapisane.";
        }
        return s;
    }
    
    /**
     * Usuwa logo firmy. Jeli plik z logo tej firmy nie istnieje - nic si nie dzieje.
     *
     * @param  id     identyfikator firmy, ktrej logo jest usuwane
     * @return true, jeli udao si usun istniejce logo
     *
     * @author ek197882
     */
    public boolean usunLogo(Integer id) {
        File f = new File(konstruujSciezkeLogo(id));
        if (f.exists()) {
            f.delete();
            return true;
        } else
            return false;
    }
    
    /**
     * Generuje ciek do katalogu z wizytwk firmy.
     *
     * @param id Identyfikator firmy
     * @return cieka
     *
     * @author ek197882
     */
    public String konstrrujHttpSciezkeKatalogVCard(Integer id) {
        return Jadro.getParametr("NIIKT_WWW") + File.separator + "vcard" + File.separator +
        id.toString().charAt(0);
    }
    
    /**
     * Generuje ciek do wizytwki firmy.
     *
     * @param id Identyfikator firmy
     * @return cieka
     *
     * @author ek197882
     */
    public String konstruujHttpSciezkeVCard(Integer id) {
        return konstrrujHttpSciezkeKatalogVCard(id)  + File.separator + id.toString() + ".vcf";
    }
    
    /**
     * Zwraca adres wizytwki.
     *
     * @param id    Identyfikator firmy
     * @return Adres wizytwki
     *
     * @author ek197882
     */
    public String adresWizytowki(Integer id) {
        return Jadro.getParametr("NIIKT_HTTP") + File.separator + "vcard" + File.separator +
        id.toString().charAt(0) + File.separator + id.toString() + ".vcf";
    }
    
    /**
     * Generuje wizytwk do pliku tekstowego, w formacie vCard. 
     * Zapisuje j w pliku okrelonym przez
     * {@link #konstruujHttpSciezkeVCard}.
     *
     * @param ogl Ogoszenie, dla ktrego tworzymy wizytwk
     * @param tel Telefon dla danego ogoszenia
     *
     * @author ek197882
     */
    public void generujWizytowke(Ogloszenie ogl, String tel) {
        File f = new File(konstruujHttpSciezkeVCard(ogl.getId()));
        File d = new File(konstrrujHttpSciezkeKatalogVCard(ogl.getId()));
        if (!d.exists())
            d.mkdirs();
        try {
            f.createNewFile();
            PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(konstruujHttpSciezkeVCard(ogl.getId()))));
            String ENC = "CHARSET=iso-8859-2";
            out.println("BEGIN:VCARD");
            out.println("N;" + ENC + ":" + ogl.getNazwa());
            if (tel != null)
                out.println("TEL:" + tel);
            out.println("ADR;" + ENC + ":;;ul." + ogl.getUlica().getNazwa() + " " + ogl.getNrDomu() + ";Warszawa;;" + ogl.getPoczta() + ";;");
            if (ogl.getFax() != null)
                out.println("FAX:" + ogl.getFax());
            if (ogl.getEmail() != null)
                out.println("EMAIL;INTERNET:" + ogl.getEmail());
            if (ogl.getStronaInternetowa() != null)
                out.println("URL:" + ogl.getStronaInternetowa());
            out.println("END:VCARD");
            out.close();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
    
    /**
     * Usuwa wizytwk. Adres / ciek do wizytwki okrela za pomoc
     * {@link #konstruujHttpSciezkeVCard}.
     *
     * @param id Identyfikator firmy
     *
     * @author ek197882
     */
    public void usunWizytowke(Integer id) {
        File f = new File(konstruujHttpSciezkeVCard(id));
        if (f.exists())
            f.delete();
    }
    
}