/*
 * KatalogOgloszen.java
 *
 * Created on 25 marzec 2004, 13:05
 */

package serwer.ogloszenia;

import net.sf.hibernate.*;
import java.util.*;
import java.awt.Point;
import java.lang.String;
import java.lang.Character;

import serwer.ogloszenia.*;
import shared.*;
import serwer.Jadro;
import serwer.mapa.*;
import serwer.Hibcio;

import java.rmi.server.UnicastRemoteObject;

/** KatalogOgloszen to jedyny obiekt umoliwiajcy wykonywanie wszelkich 
 * czynnoci administracyjnych zwizanych z ogoszeniami ({@link Ogloszenie}).
 *
 * @author  ula
 */
public class KatalogOgloszen 
    extends UnicastRemoteObject implements InterfejsOgloszenia {
    
    /** Tworzy nowy obiekt KatalogOgloszen. 
     *  @throws java.rmi.RemoteException przy bdzie inicjalizacji interfejsu RMI
     */
    public KatalogOgloszen() throws java.rmi.RemoteException {
    }
    
    
    /** Zwraca wszystkie ogoszenia znajdujce si w systemie.
     *
     *@throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     * @return lista ogosze potrzebna w celach administracyjnych
     */
    public List wszystkieOgloszenia() throws Wyjatek {
        Session s = null;
        Query q;
        List ret = new ArrayList();
        
        try {
            s = Hibcio.openSession();
            q = s.createQuery("from shared.Ogloszenie ");
            ret = q.list();
        } catch (Exception e) {
            throw new Wyjatek();
        } finally {
            Hibcio.closeSession(s);
        }
        return ret;
    }
    
    /** Odnajduje wszystkie ogoszenia, ktre zostay ostatnio dodane do systemu
     * i wymagaj akceptacji administratora. 
     *
     *@throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     * @return lista nowych ogosze 
     */
    public List noweOgloszenia() throws Wyjatek {
        Session s = null;
        Query q;
        List ret = new ArrayList();
        
        try {
            s = Hibcio.openSession();
            //wybierz niezaakceptowane
            q = s.createQuery("from shared.Ogloszenie as og " +
                              "where og.zaakceptowane = :akcept ");
            q.setParameter("akcept", new Boolean(false));
            ret = q.list();
        } catch (Exception e) {
            throw new Wyjatek();
        } finally {
            Hibcio.closeSession(s);
        }
        return ret;
    }
    
    /** Zatwierdza nowo dodane do systemu ogoszenie.
     *
     * @param ogloszenie    ogloszenie zaakceptowane przez administratora
     *@throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     */
    public void zatwierdzNoweOgloszenie(Ogloszenie ogloszenie) throws Wyjatek {
        ZarzOglosz zarz = this.zarzOglosz(ogloszenie);        
        zarz.zatwierdz();
    }
    
    /** Aktualizuje tre ogoszenia bdcego w systemie.
     *
     * @param ogloszenie    aktualizowane przez administratora ogoszenie
     *@throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     */
    public void aktualizujTrescOgloszenia(Ogloszenie ogloszenie) throws Wyjatek {
        ZarzOglosz zarz = this.zarzOglosz(ogloszenie);        
        zarz.zmienTresc(ogloszenie);
    }
    
    
    /** Odnajduje wszystkie ogoszenia, ktrym upyn termin aktualizacji, a 
     * aktualno nie zostaa potwierdzona przez firm.
     *
     * @return lista nieaktualnych ogosze 
     *@throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     */
    public List nieaktualneOgloszenia() throws Wyjatek {
        Session s = null;
        Query q;
        List ret = new ArrayList();
        Calendar c = new GregorianCalendar();
        // aktualnosc potwierdzamy co rok
        // ogloszenia, ktorym aktualnosc minela
        c.add(Calendar.YEAR, -1);
        try {
            s = Hibcio.openSession();
            q = s.createQuery("from shared.Ogloszenie as og " +
                              "where og.dataAktualizacji <= :data ");
            q.setParameter("data", c);
            ret = q.list();
        } catch (Exception e) {
            throw new Wyjatek();
        } finally {
            Hibcio.closeSession(s);
        }
        return ret;
    }
    
    /** Odnajduje wszystkie ogoszenia firm wymagajcych, ktrym upyn
     * (lub jest bliski upynicia) termin patnoci.
     *
     * @return lista nieopaconych ogosze 
     *@throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     */
    public List nieoplaconeOgloszenia() throws Wyjatek {
        Session s = null;
        Query q;
     
        List ret = new ArrayList();
        
        Calendar c = new GregorianCalendar();
        //ogloszenia ktorym data oplacenia juz minela
        c.add(Calendar.YEAR, -1);
        // jesli chcemy uwzglednic ogloszenia, ktorym data waznosci 
        // minie za 14 dni:
        // c.add(Calendar.DATE, +14);
        try {
            s = Hibcio.openSession();
            q = s.createQuery("from shared.Ogloszenie as og " +
                                   "where og.platne.dataOplacenia <= :data ");
            q.setParameter("data", c);
            ret = q.list();
        } catch (Exception e) {
            throw new Wyjatek();
        }
        finally {
            Hibcio.closeSession(s);
        }
        return ret;
    }
    
    /** Przedua termin opacenia ogoszenia patnego.
     *
     * @param platne    patne ogoszenie, ktrego termin przeduamy
     * @param doKiedy   data, do kiedy dane ogoszenie jest opacone
     *@throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     */
    public void przedluzTermin(OgloszeniePlatne platne, Date doKiedy) throws Wyjatek {
        ZarzOgloszeniePlatne zarz = this.zarzOgloszeniePlatne(platne);
        zarz.przedluzTermin(doKiedy);
    }
    
    
    /** Odnajduje zestaw ogosze speniajcych dane kryteria. Wraz z ogoszeniami
     * przesya fragment mapy (jeli uytkownik zada wyszukiwania z map) oraz
     * zestaw telefonw do kadego ogoszenia.
     *
     * @param kryteria      kryteria wyszukiwania
     * @return              lista ogosze zgodnych z kryteriami i telefonw
     *@throws Wyjatek gdy wystpi bd w dostpie do bazy danych lub podane parametry s niepoprawne
     */
    public List wyborOgloszen(Kryteria kryteria) throws Wyjatek {
        Session s = null;
        Query q;
        List alist = new ArrayList();
        List ret = new ArrayList();
        /* W drugiej poowie listy wynikw zostan zwrcone telefony do ogosze z 1. czci listy */
        List telefony = new ArrayList();
        Calendar dataAkt = new GregorianCalendar();
        dataAkt.setTime(new Date());
        //aktualnosc potwierdzamy co rok, ogloszenia musza byc aktualne i zaakceptowane
        dataAkt.add(Calendar.YEAR, -1);
        
        StringBuffer sb = new StringBuffer("from shared.Ogloszenie as og " +
                                           " where og.zaakceptowane = :akcept " +
                                           " and og.dataAktualizacji > :dataAkt ");
        
        Nip nip = null;
        Point pktBazowy = new Point();
        int promien = 0;
        List ulice = new ArrayList();
        Integer nrDomu = null;
        
        //stringi duzych liter
        String dNazwa;
        String dSlowoKluczowe;
        
        
        //w zaleznosci od tego, czy jest jakis atrybut w kryteriach czy nie
        //zbudujemy odpowiednie zapytanie HQL
        //atrybut = NULL ozn nie uwzgledniamy danego atrybutu w zapytaniach
        
        try {
            //budujemy zapytanie
            if (kryteria.getNip() != null) {
                //nip w stringu trzeba zamienic na obiekt klasy nip
                try {
                    nip = new Nip(kryteria.getNip());
                    nip.nipToInt();
                    nip.czyOsPrawna();
                } catch (Exception e) {
                    e.printStackTrace();
                    throw new Wyjatek("Niepoprawny nip.");
                }
                sb.append("and og.nip = :nip and og.osobowoscPrawna = :osp ");
            }
            if (kryteria.getNazwa() != null) {
                dNazwa = kryteria.getNazwa().toLowerCase();
                sb.append("and upper(og.nazwa) like upper('%" + dNazwa +"%') ");
            }
            if (kryteria.getSlowoKluczowe() != null) {
                dSlowoKluczowe = kryteria.getSlowoKluczowe().toLowerCase();
                sb.append("and ( upper(og.platne.slowoKluczowe1) like upper('%"+dSlowoKluczowe+"%') "
                            +" or upper(og.platne.slowoKluczowe2) like upper('%"+dSlowoKluczowe+"%') "
                            +" or upper(og.platne.slowoKluczowe3) like upper('%"+dSlowoKluczowe+"%') "
                            +" or upper(og.platne.slowoKluczowe4) like upper('%"+dSlowoKluczowe+"%') ) ");
                
            }
            
            if (kryteria.getRokZalozenia() != null) {
                //spr czy starsze czy mlodsze
                if (kryteria.getDataKiedy() == 1)
                    sb.append("and og.dataZalozenia >= :data ");
                if (kryteria.getDataKiedy() == -1)
                    sb.append("and og.dataZalozenia <= :data ");
                if (kryteria.getDataKiedy() == 0)
                    sb.append("and og.dataZalozenia = :data ");
            }
            
            
            //wyszukiwanie z ulica ma 2 wersje:
            //1. tylko dokladnie na zadanej ulicy, wtedy nrDomu = null
            //2. wokol pktu na zadanej ulicy, wtedy 
            //   - nrDomu = 0
            //     wokol srodka ulicy
            //   - nrDomu <> 0
            //     wokol zadanego pktu na tej ulicy
            if (kryteria.getUlica() != null) {
                //najpierw wyszukanie i zamiana stringa na shared.Ulica
                //TODO dialog gdy wiecej ulic
                ulice = Jadro.ulice.znajdzUlice(kryteria.getUlica());
                nrDomu = kryteria.getNr();
                
                if (ulice.size() > 0) {
                    ZarzUlica zarzUlica = Jadro.ulice.zarzUlica((shared.Ulica) ulice.get(0));
                    if (nrDomu == null) {//na ulicy
                        sb.append("and og.ulica = :ulica ");
                    }
                    else {//wokol pktu
                        pktBazowy = zarzUlica.wspolrzedneObiektu(nrDomu.intValue());
                        //nie ma takiego nru=>wyszukamy wokol srodka ulicy
                        if (pktBazowy == null)
                            pktBazowy = zarzUlica.wspolrzedneObiektu(0);
                        //zamiana promienia w m na bezwzgl odl w skali mapy
                        promien = PomocnikMapy.rzeczywisteNaBezwzgledne(kryteria.getPromien());
                        //odl z tw Pitagorasa
                        sb.append("and ((:x - og.wspolrzedna_x)*(:x - og.wspolrzedna_x)+(:y - og.wspolrzedna_y)*(:y - og.wspolrzedna_y))<=:r*:r ");
                    }
                }
                else {
                    throw new Wyjatek("Podana ulica nie istnieje.");
                }
            }
            
            if (kryteria.getBranza() != null)
                sb.append("and og.branza = :branza ");
        
            //najpierw podajemy najlepsze firmy wg rankingu
            sb.append(" order by og.ranking desc ");
         
       
            s = Hibcio.openSession();
            q = s.createQuery(sb.toString());
            
            q.setParameter("akcept", new Boolean(true));
            q.setParameter("dataAkt", dataAkt);
            
            //ustalamy parametry
            if (kryteria.getNip() != null) {
                q.setParameter("nip", new Long(nip.nipToInt()));
                q.setParameter("osp", new Boolean(nip.czyOsPrawna()));
            };
            
            if (kryteria.getRokZalozenia() != null) {
                GregorianCalendar c = new GregorianCalendar();
                c.set(kryteria.getRokZalozenia().intValue(), 0, 1);
                q.setParameter("data", c.getTime());
            };
            
            //wyszukanie wokol pktu bazowego
            if (kryteria.getUlica() != null && ulice.size() > 0) {
                if (nrDomu == null)
                    q.setParameter("ulica", (Ulica)ulice.get(0));
                else {
                    q.setParameter("x", new Double(pktBazowy.getX()));
                    q.setParameter("y", new Double(pktBazowy.getY()));
                    q.setParameter("r", new Integer(promien));
                }
            }
  
            if (kryteria.getBranza() != null)
                q.setParameter("branza", kryteria.getBranza());
            
            
            //wykonujemy zapytanie
            //wywalamy niechciane odpowiedzi
            //od razu sciagamy telefony do zadanych ogloszen
            alist = q.list();
            /* testowanie kolejnosci
            for (int i =0; i<alist.size(); i++)
                System.out.println(i + ((Ogloszenie)alist.get(i)).getNazwa() +" "+ ((Ogloszenie)alist.get(i)).getRanking());
            */
            
            int ile = kryteria.getIle();
            int zo = kryteria.getZacznijOd();
            int ss = alist.size();
            ret = new ArrayList();
            
            
            for (int i = zo; i<ss && i<zo+ile; i++) {
                Ogloszenie o = (Ogloszenie) alist.get(i);
                o.setPopularnosc(o.getPopularnosc() + 1);
                s.saveOrUpdate(o);
                ret.add(o);
                //telefon dla ogloszenia
                ArrayList t = Jadro.kontaFirm.pobierzTelefony(o.getId());
                telefony.add(t);
            };
        } catch (Exception e) {
            e.printStackTrace();
            throw new Wyjatek();
        } finally {
            Hibcio.closeSession(s);
        }
         
        
        //najlepszy fragment mapki umieszczamy na pierwszym miejscu listy
        //jak go nie chcemy to null
        if (kryteria.getCzyMapka() && ret.size() > 0) {
            FragmentMapy fr = Jadro.mapy.wybierzNajlepszyFragment(ret);
            ret.add(0, fr);
        } else
            ret.add(0, null);

        //telefony dodajemy na koniec
        ret.addAll(telefony);
        
        return ret;         
    }
    
    /** Usuwa zadane ogoszenie.
     * @param ogloszenie    ogoszenie do usunicia
     *@throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     */
    public void usunOgloszenie(Ogloszenie ogloszenie) throws Wyjatek {
        Session s = null;
        Query q;                                                                                
        Transaction tx = null;

        
        try {
            s = Hibcio.openSession();
            tx = s.beginTransaction();
            
            // usuniecie telefonow zwiazanych z tym ogloszeniem
            q = s.createQuery("from shared.Telefon as tel " +
            "where tel.ogloszenie = :ogl");
            q.setParameter("ogl", ogloszenie);
            Iterator iter = q.iterate();
            while (iter.hasNext()) {
                Telefon tel = (Telefon) iter.next();
                s.delete(tel);                                                                                
                // usuwamy ogloszenie z cache'a hibernate'a - nieelegancko,
                // ale trzeba, by potem usunac samo ogloszenie
                // wynika to z nietypowego sposobu odwzorowania telefonow i ogloszen
                s.evict(tel.getOgloszenie());
            }
            //jesli platne => usuniecie platnego i logo
            /* ogl platne jest usuwane automatem, logo w interfejsie
            if (ogloszenie.getPlatne() != null) {
                s.delete(ogloszenie.getPlatne());
                Jadro.kontaFirm.usunLogo(ogloszenie.getId());
            }
            */
            // usuwamy ogloszenie              
            s.delete(ogloszenie);
            tx.commit();
            
        } catch (Exception e) {
            if (tx != null)
                try {
                    tx.rollback();
                } catch (HibernateException he) {
                    throw new Wyjatek();
                }
            e.printStackTrace();
            throw new Wyjatek();
        } finally {
            Hibcio.closeSession(s);
        }
    }
    
    /** Zwraca obiekt zarzOglosz umoliwiajcy wykonywanie operacji na danym
     * ogoszeniu.
     * @param ogloszenie    ogoszenie, ktrym chcemy zarzdza
     *@throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     * @return              obiekt do jego zarzdzania
     */
    public ZarzOglosz zarzOglosz(Ogloszenie ogloszenie) throws Wyjatek {
        ZarzOglosz zarz = new ZarzOglosz();
        zarz.setOgloszenie(ogloszenie);
        return zarz;
    }
    
    /** Zwraca obiekt zarzOgloszPlatne umoliwiajcy wykonywanie operacji na danym
     * ogoszeniu patnym.
     * @param ogloszenie    ogoszenie patne, ktrym chcemy zarzdza
     *@throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     * @return              obiekt do jego zarzdzania
     */
    public ZarzOgloszeniePlatne zarzOgloszeniePlatne(OgloszeniePlatne ogloszenie) throws Wyjatek {
        ZarzOgloszeniePlatne zarz = new ZarzOgloszeniePlatne();
        zarz.setPlatne(ogloszenie);
        return zarz;
    }
    
    /** Zwraca ogoszenie zadanej firmy.
     * @param nip   nip firmy, ktrej ogoszenia poszukujemy
     * @return      ogoszenie tej firmy
     *@throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     */
    public Ogloszenie ogloszenieFirmy(Nip nip) throws Wyjatek {
        Session s = null;
        Query q;
        Ogloszenie o;
        
        try {
            s = Hibcio.openSession();
            //znajdz odpowiednie ogloszenie
            q = s.createQuery("from shared.Ogloszenie as og " +
                              "where og.nip = :nip " + 
                              " and og.osobowoscPrawna = :osp ");
            q.setParameter("nip", new Long(nip.nipToInt()));
            q.setParameter("osp", new Boolean(nip.czyOsPrawna()));
            o = (Ogloszenie) q.uniqueResult();
        } catch (Exception e) {       
            throw new Wyjatek();
        } finally {
            Hibcio.closeSession(s);
        }
        return o;
        
    }
  
        
    /** Zeruje licznik popularnoci dla wszystkich ogosze. Licznik z danego
     * miesica przepisuje na poprzedni miesic. Metoda powinna by wywoywana
     * co pewien ustalony okres np. miesic, aby statystyki byy wiarygodne.
     *
     *@throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     */
    public void zerujPopularnosc() throws Wyjatek {
        Session s = null;
        Query q;
        List ret = new ArrayList();
        
        try {
            s = Hibcio.openSession();
            //pobierz wszystkie ogloszenia
            q = s.createQuery("select from shared.Ogloszenie ");
            ret = q.list();
        } catch (Exception e) {
            throw new Wyjatek();
        } try {
            Transaction tn = s.beginTransaction();
            for (int i=0; i < ret.size(); i++) {
                Ogloszenie o = (Ogloszenie) ret.get(i);
                o.setPopularnoscPoprz(o.getPopularnosc());
                o.setPopularnosc(0);
                s.saveOrUpdate(o);
            };
            Hibcio.commit(s);
        } catch (Exception e) {
            try {
                Hibcio.rollback(s);
            } catch (Exception e1) {
            }
            throw new Wyjatek();
        } finally {
            Hibcio.closeSession(s);
        }
    }
    
    
    /** Zmienia podane ogoszenie patne na bezpatne.
     * @param ogloszenie    zmieniane ogoszenie
     * @throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     */
    public void zmienNaBezplatne(OgloszeniePlatne ogloszenie) throws Wyjatek {
        ZarzOgloszeniePlatne zarz = this.zarzOgloszeniePlatne(ogloszenie);
        zarz.zmienNaBezplatne();
    }
    
    /**
     * Zleca wykonanie aktualizacji.
     * @author ela
     *@param ogloszenie     aktualizowane ogoszenie
     *@throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     */
    public void potwierdzAktualnosc(Ogloszenie ogloszenie) throws Wyjatek {
        zarzOglosz(ogloszenie).potwierdzAktualnosc();
    }
}
