/*
 * InterfejsKontaFirm.java
 *
 * Created on 6 kwiecie 2004, 23:32
 */

package shared;

import java.rmi.Remote;
import java.rmi.RemoteException;
import java.util.List;
import java.util.ArrayList;

/** Sieciowy interfejs klasy {@link serwer.konto.KontaFirm}.
 *
 * @author  ula
 */
public interface InterfejsKontaFirm extends Remote {
    /** Tworzy nowe konto dla firmy na podstawie ogoszenia. Ogoszenie umieszcza
     * w bazie danych.
     * @param ogloszenie    ogoszenie, ktre firma wanie stworzya
     * @param telefony      dodatkowe telefony do firmy
     * @return napis z opisem ewentualnego bdu
     *
     * @throws Wyjatek gdy nie uda si utworzy konta dla danego ogoszenia
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     * @author ula, ek197882
     */
    public String utworzKonto(Ogloszenie ogloszenie, ArrayList telefony) throws Wyjatek,RemoteException;
    /** Sprawdza poprawno hasa.
     * @param nip   jest loginem do danego konta
     * @param haslo haso poddawane autoryzacji
     * @return ogoszenie o podanym nipie
     * @throws Wyjatek gdy nie uda si dokona autoryzacji
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     */
    public Ogloszenie autoryzacja(Nip nip, String haslo) throws Wyjatek,RemoteException;
    /** Wykonuje procedur przypominania hasa na podstawie NIP;
     * haso zostanie wysane na podany przez firm adres e-mail.
     *
     * @param nip   nip, po ktrym bdziemy identyfikowa firm
     * @throws Wyjatek jeli nie uda si odnale ogoszenia firmy w bazie
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     */
    public void przypomnijHaslo(Nip nip) throws Wyjatek,RemoteException;
    /** Zmienia haso firmy.
     * @param id        identyfikator firmy, zakadamy, e taka jest w bazie
     * @param nowe      nowe haso
     * @throws Wyjatek gdy operacje na bazie danych nie powiod si
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     */
    public void zmienHaslo(Integer id, String nowe) throws Wyjatek,RemoteException;
    /**
     * Zapisuje ogoszenie na serwer.
     *
     * @param id Identyfikator firmy, ktrej ogoszenie zostanie zapisane na serwerze
     * @param pg Plik graficzny z logo danej firmy
     * @throws Wyjatek gdy nie uda si zapisa logo na serwerze
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     * @author ek197882
     */
    public void zapiszLogoFirmy(Integer id, PlikGraficzny pg) throws Wyjatek,RemoteException;
    /**
     * Generuje nazw logo firmy.
     *
     * @author ek197882
     * @param id Identyfikator firmy, dla ktrej zostanie skonstruowana nazwa logo
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     * @return Skonstruowana cieka
     */
    public String konstruujSciezkeLogo(Integer id) throws RemoteException;
    /**
     * Generuje url typu http://.../loga/1/15 do logo firmy.
     *
     * @author ek197882
     * @param id Identyfikator firmy, dla ktrej zostanie skonstruowana nazwa logo
     * @return Skonstruowana cieka
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     */
    public String konstruujHttpSciezkeLogo(Integer id) throws RemoteException;
    /** Zwraca ogoszenie firmy na podstawie podanego id.
     * @param id    id firmy, ktrej ogoszenie naley zwrci
     * @return ogoszenie danej firmy
     * @throws Wyjatek gdy nie uda si pobra ogoszenia
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     */
    public Ogloszenie pobierzOgloszenie(Integer id) throws Wyjatek, RemoteException;
    /** Zwraca telefony zwizane z dan firm.
     * @param id    id firmy
     * @return      zbir telefonw
     * @throws Wyjatek gdy nie powiod si operacje na bazie danych
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     */
    public ArrayList pobierzTelefony(Integer id) throws Wyjatek, RemoteException;
    /**
     * Tworzy konto i zapisuje logo.
     *
     * @param ogloszenie     nowe ogoszenie
     * @param telefony       lista telefonw dla danego ogoszenia
     * @param pg             plik graficzny z logo
     * @return wynik utworzenia konta jako informacja w stringu
     * @throws Wyjatek gdy nie uda si utworzy nowego ogoszenia w bazie
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     * @author ek197882
     */
    public String utworzKontoZLogo(Ogloszenie ogloszenie, ArrayList telefony, PlikGraficzny pg) throws Wyjatek, RemoteException;
    /**
     * Usuwa logo firmy. Jeli plik z logo tej firmy nie istnieje - nic si nie dzieje.
     *
     * @param id     identyfikator firmy, ktrej logo jest usuwane
     * @return   true, jeli udao si usun istniejce logo
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     * @author ek197882
     */
    public boolean usunLogo(Integer id) throws RemoteException;
    /** Modyfikuje istniejce w bazie ogoszenie firmy.
     * @param ogloszenie    ogoszenie, ktre jest modyfikowane
     * @param telefony      dodatkowe telefony do firmy
     * @return napis z opisem ewentualnego bdu
     * @throws Wyjatek gdy nie uda si zmodyfikowa danego ogoszenia
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     * @author ula, ek197882
     */
    public String modyfikujOgloszenie(Ogloszenie ogloszenie, ArrayList telefony) throws Wyjatek, RemoteException;
    /** Modyfikuje istniejce w bazie ogoszenie firmy. Metoda dla administratora.
     * @param ogloszenie    ogoszenie, ktre jest modyfikowane
     * @param telefony      dodatkowe telefony do firmy
     * @return napis z opisem ewentualnego bdu
     * @throws Wyjatek gdy nie uda si zmodyfikowa danego ogoszenia
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     * @author ula, ek197882
     */
    public String modyfikujOgloszenieAdmin(Ogloszenie ogloszenie, ArrayList telefony) throws Wyjatek, RemoteException;
    /**
     * Generuje i zapisuje na dysku wizytwk.
     *
     * @param ogl Ogoszenie dla, ktrego tworzymy wizytwk
     * @param tel  Telefon dla danego ogoszenia
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     * @author ek197882
     */
    public void generujWizytowke(Ogloszenie ogl, String tel) throws RemoteException;
    /**
     * Usuwa wizytwk.
     *
     * @author ek197882
     * @param id Identyfikator firmy
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     */
    public void usunWizytowke(Integer id) throws RemoteException;
    /**
     * Zwraca adres wizytwki.
     *
     * @param id    Identyfikator firmy
     * @return Adres wizytwki
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     * @author ek197882
     */
    public String adresWizytowki(Integer id) throws RemoteException;
}
