/*
 * InterfejsOgloszenia.java
 *
 * Created on 6 kwiecie 2004, 23:14
 */

package shared;

import java.rmi.Remote;
import java.rmi.RemoteException;
import java.util.List;

/** Sieciowy interfejs klasy {@link serwer.ogloszenia.KatalogOgloszen}.
 *
 * @author  ula
 */
public interface InterfejsOgloszenia extends Remote {
    /** Odnajduje wszystkie ogoszenia, ktre zostay ostatnio dodane do systemu
     * i wymagaj akceptacji administratora. 
     *
     * @throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     * @return lista nowych ogosze 
     */
    public List noweOgloszenia() throws Wyjatek, RemoteException;
    /** Odnajduje wszystkie ogoszenia, ktrym upyn termin aktualizacji, a 
     * aktualno nie zostaa potwierdzona przez firm.
     *
     * @return lista nieaktualnych ogosze 
     * @throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     */
    public List nieaktualneOgloszenia() throws Wyjatek, RemoteException;
    /** Odnajduje wszystkie ogoszenia firm wymagajcych, ktrym upyn
     * (lub jest bliski upynicia) termin patnoci.
     *
     * @return lista nieopaconych ogosze 
     * @throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     */
    public List nieoplaconeOgloszenia() throws Wyjatek, RemoteException;
    /** Odnajduje zestaw ogosze speniajcych dane kryteria. Wraz z ogoszeniami
     * przesya fragment mapy (jeli uytkownik zada wyszukiwania z map) oraz
     * zestaw telefonw do kadego ogoszenia.
     *
     * @param kryteria      kryteria wyszukiwania
     * @return              lista ogosze zgodnych z kryteriami i telefonw
     * @throws Wyjatek gdy wystpi bd w dostpie do bazy danych lub podane parametry s niepoprawne
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     */
    public List wyborOgloszen(Kryteria kryteria) throws Wyjatek, RemoteException;
    /** Usuwa zadane ogoszenie.
     * @param ogloszenie    ogoszenie do usunicia
     * @throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     */
    public void usunOgloszenie(Ogloszenie ogloszenie) throws Wyjatek, RemoteException;
    /** Zmienia podane ogoszenie patne na bezpatne.
     * @param ogloszenie    zmieniane ogoszenie
     * @throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     */
    public void zmienNaBezplatne(OgloszeniePlatne ogloszenie) throws Wyjatek, RemoteException;
    /** Zwraca ogoszenie zadanej firmy.
     * @param nip   nip firmy, ktrej ogoszenia poszukujemy
     * @return      ogoszenie tej firmy
     * @throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     */
    public Ogloszenie ogloszenieFirmy(Nip nip) throws Wyjatek, RemoteException;
    /** Zeruje licznik popularnoci dla wszystkich ogosze. Licznik z danego
     * miesica przepisuje na poprzedni miesic. Metoda powinna by wywoywana
     * co pewien ustalony okres np. miesic, aby statystyki byy wiarygodne.
     *
     * @throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     */
    public void zerujPopularnosc() throws Wyjatek, RemoteException;
    /**
     * Zleca wykonanie aktualizacji.
     * @author ela
     * @param ogloszenie     aktualizowane ogoszenie
     * @throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     */
    public void potwierdzAktualnosc(Ogloszenie ogloszenie) throws Wyjatek, RemoteException;
    /** Zatwierdza nowo dodane do systemu ogoszenie.
     *
     * @param ogloszenie    ogloszenie zaakceptowane przez administratora
     * @throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     */
    public void zatwierdzNoweOgloszenie(Ogloszenie ogloszenie) throws Wyjatek, RemoteException;
    /** Aktualizuje tre ogoszenia bdcego w systemie.
     *
     * @param ogloszenie    aktualizowane przez administratora ogoszenie
     * @throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     */
    public void aktualizujTrescOgloszenia(Ogloszenie ogloszenie) throws Wyjatek, RemoteException;
    /** Zwraca wszystkie ogoszenia znajdujce si w systemie.
     *
     * @return lista ogosze potrzebna w celach administracyjnych
     * @throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     */
    public List wszystkieOgloszenia() throws Wyjatek, RemoteException;
    /** Przedua termin opacenia ogoszenia patnego.
     *
     * @param platne    patne ogoszenie, ktrego termin przeduamy
     * @param doKiedy   data, do kiedy dane ogoszenie jest opacone
     * @throws Wyjatek gdy wystpi bd w dostpie do bazy danych
     * @throws RemoteException w przypadku bdu komunikacji sieciowej
     */
    public void przedluzTermin(OgloszeniePlatne platne, java.util.Date doKiedy) throws Wyjatek, RemoteException;
}